<?php
/**
 * Renew Membership
 *
 * @package     AutomatorWP\Integrations\Paid_Memberships_Pro\Triggers\Renew_Membership
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Paid_Memberships_Pro_Renew_Membership extends AutomatorWP_Integration_Trigger {

    public $integration = 'paid_memberships_pro';
    public $trigger = 'paid_memberships_pro_renew_membership';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User renews a membership level', 'automatorwp-pro' ),
            'select_option'     => __( 'User renews <strong>a membership level</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Content title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'User renews %1$s %2$s time(s)', 'automatorwp-pro' ), '{membership}', '{times}' ),
            /* translators: %1$s: Content title. */
            'log_label'         => sprintf( __( 'User renews %1$s', 'automatorwp-pro' ), '{membership}' ),
            'action'            => array(
                'pmpro_added_order'
            ),
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'membership' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'membership',
                    'name'              => __( 'Membership Level:', 'automatorwp-pro' ),
                    'option_none_value' => 'any',
                    'option_none_label' => __( 'any membership level', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_paid_memberships_pro_get_memberships',
                    'options_cb'        => 'automatorwp_paid_memberships_pro_options_cb_membership',
                    'default'           => 'any'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_paid_memberships_pro_get_membership_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param MemberOrder $morder
     */
    public function listener( $morder ) {

        if( $morder->status !== 'success' ) {
            return;
        }

        // Check if this order was triggered already
        $triggered = get_pmpro_membership_order_meta( $morder->id, '_automatorwp_renew_membership_triggered', true );

        if ( ! empty( $triggered ) ) {
            return;
        }

        $user                = $morder->getUser();
        $membership          = $morder->getMembershipLevel();
        $user_id             = $user->ID;
        $membership_id       = $membership->id;

        // Get all active membershipships for this user
        $old_levels = pmpro_getMembershipLevelsForUser( $user_id );

        foreach ( $old_levels as $level ) {
            
            if ($level->ID !== $membership_id) {
                return;
            } 
        }

        // Bail if not is a renewal
        if( ! $morder->is_renewal() ) {
            return;
        }

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'membership_id' => $membership_id,
        ) );

        // Register that we've already triggered on this order
        update_pmpro_membership_order_meta( $morder->id, '_gamipress_renew_membership_triggered', true );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if membership is not received
        if( ! isset( $event['membership_id'] ) ) {
            return false;
        }

        $membership_id = absint( $event['membership_id'] );

        // Don't deserve if membership doesn't exists
        if( $membership_id === 0 ) {
            return false;
        }

        $required_membership_id = absint( $trigger_options['membership'] );

        // Don't deserve if membership doesn't match with the trigger option
        if( $trigger_options['membership'] !== 'any' && $membership_id !== $required_membership_id ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['membership_id'] = ( isset( $event['membership_id'] ) ? $event['membership_id'] : '' );

        return $log_meta;

    }

}

new AutomatorWP_Paid_Memberships_Pro_Renew_Membership();